<?php

namespace app\models;

use Yii;

/**
 * This is the model class for table "orders".
 *
 * @property int $id
 * @property int $user_id
 * @property int $total_price
 * @property string $payment_method
 * @property string $delivery_method
 * @property string|null $delivery_address
 * @property string|null $delivery_interval
 * @property string|null $status
 * @property string $created_at
 *
 * @property OrderItem[] $orderItems
 * @property User $user
 */
class Order extends \yii\db\ActiveRecord
{
    /**
     * ENUM field values
     */
    const PAYMENT_METHOD_CASH = 'cash';
    const PAYMENT_METHOD_CASHLESS = 'cashless';
    const DELIVERY_METHOD_PICKUP = 'pickup';
    const DELIVERY_METHOD_COURIER = 'courier';

    // Status constants with unique names
    const STATUS_NEW = 'Новый';
    const STATUS_CONFIRMED = 'Подтверждён';
    const STATUS_IN_PROCESS = 'В процессе';
    const STATUS_DELIVERED = 'Доставлен';
    const STATUS_CANCELLED = 'Отменён';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'orders';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['delivery_address', 'delivery_interval'], 'default', 'value' => null],
            [['status'], 'default', 'value' => self::STATUS_NEW],
            [['user_id', 'total_price', 'payment_method', 'delivery_method'], 'required'],
            [['user_id', 'total_price'], 'integer'],
            [['payment_method', 'delivery_method', 'delivery_address', 'status'], 'string'],
            [['created_at'], 'safe'],
            [['delivery_interval'], 'string', 'max' => 255],
            ['payment_method', 'in', 'range' => array_keys(self::optsPaymentMethod())],
            ['delivery_method', 'in', 'range' => array_keys(self::optsDeliveryMethod())],
            ['status', 'in', 'range' => array_keys(self::optsStatus())],
            [['user_id'], 'exist', 'skipOnError' => true, 'targetClass' => User::class, 'targetAttribute' => ['user_id' => 'id']],

            // Правила для формы
            [['delivery_address'], 'required', 'when' => function($model) {
                return $model->delivery_method === self::DELIVERY_METHOD_COURIER;
            }, 'whenClient' => "function(attribute, value) {
                return $('#order-delivery_method').val() === '" . self::DELIVERY_METHOD_COURIER . "';
            }"],

            [['delivery_interval'], 'required', 'when' => function($model) {
                return $model->delivery_method === self::DELIVERY_METHOD_COURIER;
            }, 'whenClient' => "function(attribute, value) {
                return $('#order-delivery_method').val() === '" . self::DELIVERY_METHOD_COURIER . "';
            }"],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'Пользователь',
            'total_price' => 'Итоговая цена',
            'payment_method' => 'Способ оплаты',
            'delivery_method' => 'Способ доставки',
            'delivery_address' => 'Адрес доставки',
            'delivery_interval' => 'Интервал доставки',
            'status' => 'Статус',
            'created_at' => 'Дата создания',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->created_at = date('Y-m-d H:i:s');
            }
            return true;
        }
        return false;
    }

    /**
     * Gets query for [[OrderItem]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getOrderItems()
    {
        return $this->hasMany(OrderItem::class, ['order_id' => 'id']);
    }

    /**
     * Gets query for [[User]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    /**
     * column payment_method ENUM value labels
     * @return string[]
     */
    public static function optsPaymentMethod()
    {
        return [
            self::PAYMENT_METHOD_CASH => 'Наличные',
            self::PAYMENT_METHOD_CASHLESS => 'Безналичный расчет',
        ];
    }

    /**
     * column delivery_method ENUM value labels
     * @return string[]
     */
    public static function optsDeliveryMethod()
    {
        return [
            self::DELIVERY_METHOD_PICKUP => 'Самовывоз',
            self::DELIVERY_METHOD_COURIER => 'Доставка курьером (+300 руб.)',
        ];
    }

    /**
     * column status ENUM value labels
     * @return string[]
     */
    public static function optsStatus()
    {
        return [
            self::STATUS_NEW => 'Новый',
            self::STATUS_CONFIRMED => 'Подтверждён',
            self::STATUS_IN_PROCESS => 'В процессе',
            self::STATUS_DELIVERED => 'Доставлен',
            self::STATUS_CANCELLED => 'Отменён',
        ];
    }

    /**
     * @return string
     */
    public function displayPaymentMethod()
    {
        return self::optsPaymentMethod()[$this->payment_method] ?? $this->payment_method;
    }

    /**
     * @return bool
     */
    public function isPaymentMethodCash()
    {
        return $this->payment_method === self::PAYMENT_METHOD_CASH;
    }

    public function setPaymentMethodToCash()
    {
        $this->payment_method = self::PAYMENT_METHOD_CASH;
    }

    /**
     * @return bool
     */
    public function isPaymentMethodCashless()
    {
        return $this->payment_method === self::PAYMENT_METHOD_CASHLESS;
    }

    public function setPaymentMethodToCashless()
    {
        $this->payment_method = self::PAYMENT_METHOD_CASHLESS;
    }

    /**
     * @return string
     */
    public function displayDeliveryMethod()
    {
        return self::optsDeliveryMethod()[$this->delivery_method] ?? $this->delivery_method;
    }

    /**
     * @return bool
     */
    public function isDeliveryMethodPickup()
    {
        return $this->delivery_method === self::DELIVERY_METHOD_PICKUP;
    }

    public function setDeliveryMethodToPickup()
    {
        $this->delivery_method = self::DELIVERY_METHOD_PICKUP;
    }

    /**
     * @return bool
     */
    public function isDeliveryMethodCourier()
    {
        return $this->delivery_method === self::DELIVERY_METHOD_COURIER;
    }

    public function setDeliveryMethodToCourier()
    {
        $this->delivery_method = self::DELIVERY_METHOD_COURIER;
    }

    /**
     * @return string
     */
    public function displayStatus()
    {
        return self::optsStatus()[$this->status] ?? $this->status;
    }

    /**
     * @return bool
     */
    public function isStatusNew()
    {
        return $this->status === self::STATUS_NEW;
    }

    public function setStatusToNew()
    {
        $this->status = self::STATUS_NEW;
    }

    /**
     * @return bool
     */
    public function isStatusConfirmed()
    {
        return $this->status === self::STATUS_CONFIRMED;
    }

    public function setStatusToConfirmed()
    {
        $this->status = self::STATUS_CONFIRMED;
    }

    /**
     * @return bool
     */
    public function isStatusInProcess()
    {
        return $this->status === self::STATUS_IN_PROCESS;
    }

    public function setStatusToInProcess()
    {
        $this->status = self::STATUS_IN_PROCESS;
    }

    /**
     * @return bool
     */
    public function isStatusDelivered()
    {
        return $this->status === self::STATUS_DELIVERED;
    }

    public function setStatusToDelivered()
    {
        $this->status = self::STATUS_DELIVERED;
    }

    /**
     * @return bool
     */
    public function isStatusCancelled()
    {
        return $this->status === self::STATUS_CANCELLED;
    }

    public function setStatusToCancelled()
    {
        $this->status = self::STATUS_CANCELLED;
    }

    /**
     * Получить общее количество товаров в заказе
     * @return int
     */
    public function getTotalQuantity()
    {
        $total = 0;
        foreach ($this->orderItems as $item) {
            $total += $item->quantity;
        }
        return $total;
    }

    /**
     * Проверка возможности отмены заказа
     * @return bool
     */
    public function canBeCancelled()
    {
        return $this->isStatusNew() || $this->isStatusConfirmed();
    }
}